﻿namespace Hims.Api.Controllers
{
    using System;
    using System.Collections.Generic;
    using System.Linq;
    using System.Threading.Tasks;
    using Domain.Services;
    using Hims.Api.Models;
    using Microsoft.AspNetCore.Authorization;
    using Microsoft.AspNetCore.Mvc;
    using Microsoft.CodeAnalysis.CSharp;
    using Npgsql;
    using Shared.DataFilters;
    using Shared.EntityModels;
    using Shared.Library.Enums;
    using Shared.UserModels.Filters;
    using Utilities;

    /// <inheritdoc />
    [Authorize]
    [Route("api/payCategories")]
    [Consumes("application/json")]
    [Produces("application/json")]
    public class PayCategoryController : BaseController
    {
        /// <summary>
        /// The payType services.
        /// </summary>
        private readonly IPayCategoryService payServices;

        /// <summary>
        /// The audit log services.
        /// </summary>
        private readonly IAuditLogService auditLogServices;

        /// <inheritdoc />
        public PayCategoryController(IPayCategoryService payServices, IAuditLogService auditLogServices)
        {
            this.payServices = payServices;
            this.auditLogServices = auditLogServices;
        }

        /// <summary>
        /// The fetch paytypes.
        /// </summary>
        /// <returns>
        /// The list of paytypes.
        /// </returns>
        /// <remarks>
        /// ### REMARKS ###
        /// The following codes are returned
        /// - 200 - List of paytypes.
        /// - 500 - Problem with Server side code.
        /// </remarks>
        [HttpPost]
        [Route("fetch")]
        public async Task<ActionResult> FetchAsync([FromBody] PayCategoryModel model)
        {
            model = (PayCategoryModel)EmptyFilter.Handler(model);
            var response = await this.payServices.FetchAsync(model);
            return response == null ? this.ServerError() : this.Success(response);
        }
       
        /// <summary>
        /// Modifies the paytype status asynchronous.
       /// </summary>
        /// <param name = "model" > The model.</param>
        /// <returns></returns>
        [HttpPost]
        [Route("modify")]
        public async Task<ActionResult> ModifyPayTypeStatusAsync([FromBody] PayCategoryModel model, [FromHeader] LocationHeader header)
        {
            model = (PayCategoryModel)EmptyFilter.Handler(model);
            model.LocationId = (int)(!string.IsNullOrEmpty(header.LocationId) ? int.Parse(header.LocationId) : (int?)null);
            var response = await this.payServices.ActivateOrDeactivatePayType(model);
            
         

                var auditLog = new AuditLogModel
                {
                    AccountId = model.ModifiedBy,
                    LogTypeId = (int)LogTypes.PayCategory,
                    LogFrom = short.Parse(model.LoginRoleId.ToString()),
                    LogDate = DateTime.UtcNow.AddMinutes(330),
                    LogDescription = $@"<b>{model.ModifiedByName}</b> has <b>{((bool)model.Active ? "Activated" : "Deactivated")}</b> the PayCategory <b> {model.PayCategoryName}</b>  successfully",
                    LocationId = Convert.ToInt32(header.LocationId)
                };
                await this.auditLogServices.LogAsync(auditLog);
            
            return this.Success(response);
        }

        /// <summary>
        /// The create async.
        /// </summary>
        /// <param name="model">
        /// The model.
        /// </param>
        /// <returns>
        /// The <see cref="Task"/>.
        /// </returns>
        [HttpPost]
        [Route("insert")]
        public async Task<ActionResult> CreateAsync([FromBody] PayCategoryModel model, [FromHeader] LocationHeader header)
        {
            model = (PayCategoryModel)EmptyFilter.Handler(model);
            model.LocationId = (int)(!string.IsNullOrEmpty(header.LocationId) ? int.Parse(header.LocationId) : (int?)null);
            var response = await this.payServices.InsertAsync(model);
            switch (response)
            {
                case -1:
                    return this.Conflict("Given Pay Category has already been exists with us.");
                case 0:
                    return this.ServerError();
            }
            try
            {
                if (response > 0)
                {
                    var auditLog = new AuditLogModel
                    {
                        AccountId = model.CreatedBy,
                        LogTypeId = (int)LogTypes.PayCategory,
                        LogDate = DateTime.UtcNow.AddMinutes(330),
                        LogFrom = short.Parse(model.LoginRoleId.ToString()),
                        LocationId = (int)model.LocationId,
                        LogDescription = $"<b>{model.CreatedByName}</b> <b> has created </b>  the <b> New Pay Category</b> of <strong> {model.PayCategoryName}</strong> successfully."
                    };
                    await this.auditLogServices.LogAsync(auditLog);
                }

            }
            catch (Exception e) {
                return null;
            }
            return  this.Success(response);         
        }

        /// <summary>
        /// The update async.
        /// </summary>
        /// <param name="model"></param>
        /// <returns>
        /// The <see cref="Task"/>.
        /// </returns>
        [HttpPost]
        [Route("update")]
        public async Task<ActionResult> UpdateAsync([FromBody] PayCategoryModel model, [FromHeader] LocationHeader header)
        {
            model = (PayCategoryModel)EmptyFilter.Handler(model);
            model.LocationId = (int)(!string.IsNullOrEmpty(header.LocationId) ? int.Parse(header.LocationId) : (int?)null);
            var response = await this.payServices.UpdateAsync(model);
            switch (response)
            {
                case -1:
                    return this.Conflict("Given Pay Category has already been exists with us.");
                case 0:
                    return this.ServerError();
            }
            try
            {

                if (response > 0)
                {
                    var auditLog = new AuditLogModel
                    {
                        AccountId = model.ModifiedBy,
                        LogTypeId = (int)LogTypes.PayCategory,
                        LogDate = DateTime.UtcNow.AddMinutes(330),
                        LogFrom = short.Parse(model.LoginRoleId.ToString()),
                        LocationId = (int)model.LocationId,
                        LogDescription = $"<b>{model.ModifiedByName}</b> has<b> Updated</b>  the <b>Pay Category</b> of <strong> {model.PayCategoryName}</strong> successfully."
                    };
                    await this.auditLogServices.LogAsync(auditLog);
                }
            }
            catch (Exception e) { return null; }
            return this.Success(response);
        }

    }

}